<?php
// --- Bagian 1: Konfigurasi dan Fungsi PHP Inti ---

// AKTIFKAN SESSION untuk melacak status login
session_start();

// KONFIGURASI KEAMANAN: Kata sandi Admin. 
define('ADMIN_PASSWORD', 'Mifwan.id01'); 

// Pengaturan
$data_file = 'data.json';
$days_of_week = ['Min', 'Sen', 'Sel', 'Rab', 'Kam', 'Jum', 'Sab'];
$status_options = ['H' => 'Hadir', 'S' => 'Sakit', 'I' => 'Izin', 'A' => 'Alfa'];
$is_public_view = isset($_GET['view']) && $_GET['view'] === 'public'; // Mode Viewer/Publik

// Cek status Login Admin
$is_admin_logged_in = isset($_SESSION['is_admin_logged_in']) && $_SESSION['is_admin_logged_in'] === true;

// Tentukan bulan saat ini
$current_date = new DateTime();
$current_month_year = $_GET['month'] ?? $current_date->format('Y-m'); 
$current_month_name = getIndonesianMonthName($current_month_year);

// Fungsi untuk menangani data file
function loadData($file) {
    if (!file_exists($file) || filesize($file) == 0) {
        return ['classList' => [], 'classData' => []];
    }
    $data = file_get_contents($file);
    return json_decode($data, true) ?: ['classList' => [], 'classData' => []];
}

function saveData($file, $data) {
    // Hanya simpan data jika bukan dalam mode publik (read-only) DAN admin sudah login
    global $is_public_view, $is_admin_logged_in;
    if (!$is_public_view && $is_admin_logged_in) {
        return file_put_contents($file, json_encode($data, JSON_PRETTY_PRINT));
    }
    return false;
}

function getIndonesianMonthName($month_year) {
    $parts = explode('-', $month_year);
    if (count($parts) < 2) return $month_year;
    $month_index = (int)$parts[1];
    $year = $parts[0];
    $months = [
        1 => 'Januari', 2 => 'Februari', 3 => 'Maret', 4 => 'April', 5 => 'Mei', 6 => 'Juni',
        7 => 'Juli', 8 => 'Agustus', 9 => 'September', 10 => 'Oktober', 11 => 'November', 12 => 'Desember'
    ];
    return ($months[$month_index] ?? '') . ' ' . $year;
}

// Fungsi untuk mendapatkan kelas CSS dari status
function getStatusClass($status, $is_public_view) {
    // Kelas yang lebih minimalis dan profesional
    $base = $is_public_view ? 'px-3 py-1 font-bold rounded-md' : 'attendance-select';
    switch ($status) {
        case 'H': return "H bg-green-100 text-green-700 $base";
        case 'S': return "S bg-red-100 text-red-700 $base";
        case 'I': return "I bg-yellow-100 text-yellow-700 $base";
        case 'A': default: return "A bg-gray-100 text-gray-700 $base";
    }
}


// --- Bagian 2: Penanganan Otentikasi dan Data POST/AJAX (Admin Only) ---

$login_error = '';

// 2A. Menangani Logout (jika ada)
if (isset($_GET['action']) && $_GET['action'] === 'logout') {
    session_destroy();
    header('Location: ' . strtok($_SERVER["REQUEST_URI"], '?'));
    exit;
}

// 2B. Menangani Permintaan Login POST
if (isset($_POST['action']) && $_POST['action'] === 'login') {
    $password = $_POST['password'] ?? '';
    if ($password === ADMIN_PASSWORD) {
        $_SESSION['is_admin_logged_in'] = true;
        header('Location: ' . strtok($_SERVER["REQUEST_URI"], '?'));
        exit;
    } else {
        $login_error = 'Kata sandi salah. Silakan coba lagi.';
    }
}


// Aksi POST/AJAX (Hanya diizinkan jika admin sudah login)
if (!$is_public_view && $is_admin_logged_in && $_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] !== 'login') {
    header('Content-Type: application/json');
    $response = ['success' => false, 'message' => 'Gagal: Tindakan tidak valid.'];

    $server_data = loadData($data_file);
    $current_class_id = $_POST['class_id'] ?? $_REQUEST['class_id'] ?? ($server_data['classList'][0]['id'] ?? null);

    switch ($_POST['action']) {
        case 'add_class':
            $newName = trim($_POST['new_name'] ?? '');
            if (!$newName) { $response['message'] = 'Nama kelas tidak boleh kosong.'; break; }
            $newId = strtolower(preg_replace('/[^\w-]/', '', str_replace(' ', '_', $newName)));
            if (empty($newId) || array_filter($server_data['classList'], fn($c) => $c['id'] === $newId)) {
                 $newId = $newId . '_' . time();
            }

            $server_data['classList'][] = ['id' => $newId, 'name' => $newName];
            $server_data['classData'][$newId] = ['students' => [], 'attendance_records' => []];
            $response = ['success' => true, 'message' => "Kelas '{$newName}' berhasil ditambahkan.", 'new_id' => $newId];
            break;

        case 'delete_class':
            $password_check = $_POST['password_check'] ?? '';
            $classId = $_POST['class_id'] ?? null;
            
            if ($password_check !== ADMIN_PASSWORD) {
                $response['message'] = 'Kata sandi admin salah. Penghapusan dibatalkan.';
                break;
            }

            if ($classId && isset($server_data['classData'][$classId])) {
                $server_data['classList'] = array_values(array_filter($server_data['classList'], fn($c) => $c['id'] !== $classId));
                unset($server_data['classData'][$classId]);
                $next_class_id = $server_data['classList'][0]['id'] ?? null;
                $response = ['success' => true, 'message' => "Kelas berhasil dihapus.", 'new_id' => $next_class_id];
            } else {
                $response['message'] = 'Kelas tidak ditemukan atau ID tidak valid.';
            }
            break;

        case 'update_attendance':
            $studentId = $_POST['student_id'] ?? '';
            $date = $_POST['date'] ?? '';
            $status = $_POST['status'] ?? 'A';
            $classId = $_POST['class_id'] ?? $current_class_id;

            if ($classId && $studentId && $date && isset($server_data['classData'][$classId])) {
                if (!isset($server_data['classData'][$classId]['attendance_records'][$current_month_year])) {
                    $server_data['classData'][$classId]['attendance_records'][$current_month_year] = [];
                }
                if (!isset($server_data['classData'][$classId]['attendance_records'][$current_month_year][$studentId])) {
                    $server_data['classData'][$classId]['attendance_records'][$current_month_year][$studentId] = [];
                }
                
                $server_data['classData'][$classId]['attendance_records'][$current_month_year][$studentId][$date] = $status;
                $response = ['success' => true, 'message' => 'Kehadiran berhasil diperbarui.'];
            } else {
                $response['message'] = 'Data tidak lengkap atau kelas tidak ditemukan.';
            }
            break;

        case 'add_student':
            $name = trim($_POST['name'] ?? '');
            $gender = $_POST['gender'] ?? 'L';
            $idNumber = $_POST['id_number'] ?? '';
            $classId = $_POST['class_id'] ?? $current_class_id;

            if ($classId && $name && isset($server_data['classData'][$classId])) {
                $students = $server_data['classData'][$classId]['students'] ?? [];
                
                $next_id_num = 1;
                foreach ($students as $s) {
                    if (preg_match('/^s(\d+)$/', $s['id'], $matches)) {
                        $next_id_num = max($next_id_num, (int)$matches[1] + 1);
                    }
                }
                
                $newId = 's' . $next_id_num;
                $newStudent = ['id' => $newId, 'name' => $name, 'gender' => $gender, 'id_number' => $idNumber];
                
                $server_data['classData'][$classId]['students'][] = $newStudent;
                $response = ['success' => true, 'message' => "Siswa '{$name}' berhasil ditambahkan."];
            } else {
                $response['message'] = 'Nama siswa atau kelas tidak valid.';
            }
            break;

        case 'save_students_json':
            $jsonInput = $_POST['json_data'] ?? '';
            $classId = $_POST['class_id'] ?? $current_class_id;

            try {
                $newStudents = json_decode($jsonInput, true);
                if (!is_array($newStudents)) throw new Exception("Format JSON tidak valid.");
                
                foreach ($newStudents as $s) {
                    if (!isset($s['id']) || !isset($s['name'])) throw new Exception("Setiap siswa harus memiliki 'id' dan 'name'.");
                }
                
                $server_data['classData'][$classId]['students'] = $newStudents;
                $response = ['success' => true, 'message' => 'Daftar siswa berhasil disimpan dari JSON.'];
            } catch (Exception $e) {
                $response['message'] = 'Gagal memproses JSON: ' . $e->getMessage();
            }
            break;
    }

    if ($response['success']) {
        saveData($data_file, $server_data);
    }
    
    echo json_encode($response);
    exit;
} // Penutup blok 'if' untuk Aksi POST/AJAX


// --- Bagian 3: Persiapan Data untuk HTML Rendering ---
$server_data = loadData($data_file);
$current_class_id = $_REQUEST['class_id'] ?? ($server_data['classList'][0]['id'] ?? null);
$current_class_data = $server_data['classData'][$current_class_id] ?? ['students' => [], 'attendance_records' => []];
$students = $current_class_data['students'] ?? [];
$attendance_data_month = $current_class_data['attendance_records'][$current_month_year] ?? [];

$current_class_name = 'Pilih Kelas';
foreach ($server_data['classList'] as $class) {
    if ($class['id'] === $current_class_id) {
        $current_class_name = $class['name'];
        break;
    }
}

// Menghitung hari dalam bulan yang dipilih
$days_in_month = [];
if ($current_class_id) {
    $day_iterator = new DateTime($current_month_year . '-01');
    $end_of_month = new DateTime($day_iterator->format('Y-m-t'));
    while ($day_iterator <= $end_of_month) {
        $days_in_month[] = [
            'full' => $day_iterator->format('Y-m-d'),
            'day' => $day_iterator->format('j'),
            'dayOfWeek' => (int)$day_iterator->format('w') // 0=Minggu, 6=Sabtu
        ];
        $day_iterator->modify('+1 day');
    }
}
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Daftar Hadir Siswa - Profesional V4</title>
    <!-- Load Tailwind CSS -->
    <script src="https://cdn.tailwindcss.com"></script>
    <style>
        :root { font-family: 'Inter', sans-serif; }
        /* Styling untuk mode Admin */
        .attendance-select {
            width: 100%;
            text-align: center;
            padding: 0.125rem;
            font-size: 0.75rem;
            font-weight: 600;
            border-radius: 0.375rem;
            transition: background-color 0.15s;
            border: 0;
            cursor: pointer;
            appearance: none; 
            background-image: none;
        }
        /* Color styling */
        .H { color: #065f46; } 
        .S { color: #991b1b; } 
        .I { color: #a16207; } 
        .A { color: #4b5563; } 
        
        /* Sticky column for student names */
        .sticky-col {
            position: sticky;
            left: 0;
            z-index: 20; 
            background-color: inherit; 
            box-shadow: 2px 0 5px -2px rgba(0, 0, 0, 0.1); 
        }
        /* Custom scrollbar for better aesthetics */
        ::-webkit-scrollbar {
            height: 8px;
        }
        ::-webkit-scrollbar-thumb {
            background: #cbd5e1;
            border-radius: 10px;
        }
        ::-webkit-scrollbar-track {
            background: #f1f5f9;
        }
    </style>
</head>
<body class="bg-gray-50 min-h-screen p-4 sm:p-8">

    <!-- Status Pesan Server/AJAX (Fixed and less intrusive) -->
    <?php if (!$is_public_view && $is_admin_logged_in): ?>
    <div id="server-status" class="fixed top-0 left-1/2 -translate-x-1/2 w-11/12 max-w-lg mt-4 p-3 font-medium rounded-lg border hidden transition duration-500 ease-in-out z-50 shadow-xl">
    </div>
    <?php endif; ?>

    <div class="max-w-7xl mx-auto" id="main-content">
        
        <header class="text-center mb-8 bg-white p-6 rounded-xl shadow-lg">
            <h1 class="text-3xl font-extrabold text-gray-900">
                Sistem Daftar Hadir Siswa
            </h1>
            <p class="text-sm font-semibold mt-1">
                Akses: <span class="
                    <?= $is_public_view ? 'text-blue-600' : ($is_admin_logged_in ? 'text-indigo-600' : 'text-red-600') ?>">
                    <?php 
                        if ($is_public_view) echo 'Publik (Viewer)';
                        else if ($is_admin_logged_in) echo 'Administrator';
                        else echo 'Administrator (Belum Login)';
                    ?>
                </span>
            </p>
            <!-- Tombol Navigasi Mode -->
            <div class="mt-4 flex justify-center space-x-3">
                <?php if ($is_public_view): ?>
                    <a href="<?= htmlspecialchars(preg_replace('/&?view=public/', '', $_SERVER['REQUEST_URI'] ?? '')) ?>" class="px-3 py-1.5 bg-indigo-600 text-white font-medium rounded-lg hover:bg-indigo-700 transition duration-150 text-sm">
                        <span class="inline-block align-middle">Mode Admin</span>
                    </a>
                <?php elseif ($is_admin_logged_in): ?>
                    <a href="<?= htmlspecialchars((isset($_SERVER['REQUEST_URI']) ? $_SERVER['REQUEST_URI'] : 'attendance_app.php') . (strpos(($_SERVER['REQUEST_URI'] ?? ''), '?') === false ? '?' : '&') . 'view=public') ?>" class="px-3 py-1.5 bg-blue-600 text-white font-medium rounded-lg hover:bg-blue-700 transition duration-150 text-sm">
                        <span class="inline-block align-middle">Mode Publik</span>
                    </a>
                    <a href="?action=logout" class="px-3 py-1.5 bg-gray-500 text-white font-medium rounded-lg hover:bg-gray-600 transition duration-150 text-sm">
                        <span class="inline-block align-middle">Logout</span>
                    </a>
                <?php endif; ?>
            </div>
        </header>

        <!-- KONDISI LOGIN -->
        <?php if (!$is_public_view && !$is_admin_logged_in): ?>
            <div class="max-w-md mx-auto p-8 bg-white shadow-2xl rounded-xl border border-red-200 mt-10">
                <h2 class="text-2xl font-bold text-red-700 mb-4">Akses Administrator</h2>
                <?php if ($login_error): ?>
                    <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded-lg relative mb-4 text-sm" role="alert">
                        <strong class="font-bold">Gagal:</strong>
                        <span class="block sm:inline"><?= htmlspecialchars($login_error) ?></span>
                    </div>
                <?php endif; ?>
                <form method="POST" action="">
                    <input type="hidden" name="action" value="login">
                    <div class="mb-4">
                        <label for="password" class="block text-sm font-medium text-gray-700 mb-1">Kata Sandi</label>
                        <input type="password" id="password" name="password" required 
                               class="w-full p-3 border border-gray-300 rounded-lg focus:ring-red-500 focus:border-red-500 shadow-sm" 
                               placeholder="Masukan Password">
                    </div>
                    <button type="submit" class="w-full px-4 py-3 bg-red-600 text-white font-semibold rounded-lg hover:bg-red-700 transition duration-150 shadow-md">
                        Masuk
                    </button>
                </form>
            </div>
        <?php else: ?>

        <!-- Pilihan Kelas & Bulan -->
        <div class="mb-8 p-6 bg-white shadow-lg rounded-xl border border-gray-200">
            <h2 class="text-xl font-bold mb-3 text-gray-700">Pilih Data</h2>
            <form method="GET" class="flex flex-col sm:flex-row gap-4 items-end">
                <?php if ($is_public_view): ?>
                    <input type="hidden" name="view" value="public">
                <?php endif; ?>
                <div class="w-full sm:w-1/2">
                    <label for="class-selector" class="block text-sm font-medium text-gray-700 mb-1">Kelas:</label>
                    <select id="class-selector" name="class_id" onchange="this.form.submit()" class="w-full p-3 border border-gray-300 rounded-lg focus:ring-indigo-500 focus:border-indigo-500 bg-white shadow-sm text-sm">
                        <?php if (empty($server_data['classList'])): ?>
                            <option value="">-- Tidak Ada Kelas --</option>
                        <?php endif; ?>
                        <?php foreach ($server_data['classList'] as $class): ?>
                            <option value="<?= htmlspecialchars($class['id']) ?>" <?= $current_class_id === $class['id'] ? 'selected' : '' ?>>
                                <?= htmlspecialchars($class['name']) ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="w-full sm:w-1/2">
                    <label for="month-selector" class="block text-sm font-medium text-gray-700 mb-1">Bulan:</label>
                    <input type="month" id="month-selector" name="month" value="<?= htmlspecialchars($current_month_year) ?>" 
                           onchange="this.form.submit()" 
                           class="w-full p-3 border border-gray-300 rounded-lg focus:ring-indigo-500 focus:border-indigo-500 bg-white shadow-sm text-sm">
                </div>
            </form>
            <p class="text-sm text-gray-500 mt-3 font-medium">Data saat ini: <span class="font-bold text-indigo-600"><?= htmlspecialchars($current_class_name) ?></span> untuk <span class="font-bold text-indigo-600"><?= htmlspecialchars($current_month_name) ?></span></p>

            <!-- Bagian Administrasi Kelas dan Siswa (Hanya Admin) -->
            <?php if (!$is_public_view && $is_admin_logged_in): ?>
                <div class="mt-6 border-t pt-4">
                    <button id="toggle-admin-panel" class="w-full flex items-center justify-center space-x-2 px-4 py-2 bg-indigo-500 text-white font-semibold rounded-lg hover:bg-indigo-600 transition shadow-md">
                        <svg id="arrow-icon" class="w-4 h-4 transform transition-transform duration-300" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 9l-7 7-7-7"></path></svg>
                        <span>Kelola Kelas & Siswa</span>
                    </button>
                    
                    <div id="admin-panel" class="hidden mt-4 space-y-6 p-4 border border-indigo-200 rounded-lg bg-indigo-50">
                        
                        <!-- 1. Tambah Kelas -->
                        <div class="bg-white p-4 rounded-lg shadow-inner">
                            <h3 class="font-bold text-lg text-indigo-700 mb-3 border-b pb-2">Tambah Kelas Baru</h3>
                            <div class="flex flex-col sm:flex-row gap-3">
                                <input type="text" id="new-class-name" class="flex-grow p-2 border border-gray-300 rounded-md text-sm shadow-sm" placeholder="Contoh: Kelas 6B">
                                <button id="add-class-btn" class="px-4 py-2 bg-green-600 text-white font-semibold rounded-md hover:bg-green-700 transition text-sm">
                                    Simpan Kelas
                                </button>
                            </div>
                        </div>

                        <!-- 2. Kelola Siswa (Form Tambah Tunggal & JSON) -->
                        <?php if ($current_class_id): ?>
                        <div class="bg-white p-4 rounded-lg shadow-inner">
                            <h3 class="font-bold text-lg text-indigo-700 mb-4 border-b pb-2">Manajemen Siswa (<?= htmlspecialchars($current_class_name) ?>)</h3>
                            
                            <!-- Tambah Siswa Tunggal -->
                            <div class="mb-4">
                                <h4 class="font-semibold text-gray-700 mb-2">Tambah Siswa Tunggal</h4>
                                <div class="grid grid-cols-2 md:grid-cols-4 gap-3">
                                    <input type="text" id="new-student-name" class="p-2 border border-gray-300 rounded-md text-sm shadow-sm col-span-2 md:col-span-1" placeholder="Nama Siswa">
                                    <select id="new-student-gender" class="p-2 border border-gray-300 rounded-md text-sm bg-white shadow-sm">
                                        <option value="L">Laki-laki (L)</option>
                                        <option value="P">Perempuan (P)</option>
                                    </select>
                                    <input type="text" id="new-student-id-number" class="p-2 border border-gray-300 rounded-md text-sm shadow-sm" placeholder="NISN/NIS">
                                    <button id="add-student-btn" class="px-4 py-2 bg-indigo-600 text-white font-semibold rounded-md hover:bg-indigo-700 transition text-sm">Tambah</button>
                                </div>
                            </div>

                            <!-- Kelola Siswa Massal JSON -->
                            <div>
                                <h4 class="font-semibold text-gray-700 mb-2 mt-4 border-t pt-3">Kelola Siswa Massal (JSON)</h4>
                                <textarea id="student-json-input" rows="5" class="w-full p-2 border border-gray-300 rounded-lg focus:ring-indigo-500 font-mono text-xs shadow-inner" 
                                    placeholder='[{"id": "s001", "name": "Budi Santoso", "gender": "L", "id_number": "123"}, {"id": "s002", "name": "Ani Wijaya", "gender": "P", "id_number": "456"}]'><?= htmlspecialchars(json_encode($students, JSON_PRETTY_PRINT)) ?></textarea>
                                <button id="save-students-btn" class="mt-2 w-full px-4 py-2 bg-purple-600 text-white font-semibold rounded-md hover:bg-purple-700 transition text-sm">
                                    Simpan Daftar Siswa JSON
                                </button>
                            </div>
                        </div>
                        
                        <!-- 3. Hapus Kelas (Aksi Destruktif) -->
                        <div class="bg-red-50 p-4 rounded-lg border border-red-300 shadow-inner">
                            <h3 class="font-bold text-lg text-red-700 mb-3 border-b border-red-200 pb-2">Hapus Kelas (Aksi Berbahaya)</h3>
                            <div class="flex flex-col sm:flex-row gap-3">
                                <button id="show-delete-area-btn" class="flex-grow px-4 py-2 bg-red-600 text-white font-semibold rounded-md hover:bg-red-700 transition text-sm">
                                    Hapus Kelas <?= htmlspecialchars($current_class_name) ?>
                                </button>
                                <!-- Area Verifikasi Password (Hidden by default) -->
                                <div id="delete-verification-area" class="hidden w-full sm:w-auto flex flex-col sm:flex-row gap-2">
                                    <input type="password" id="delete-password-input" class="p-2 border border-red-400 rounded-md text-sm flex-grow" placeholder="Password Admin">
                                    <button id="confirm-delete-btn" class="px-4 py-2 bg-red-700 text-white font-semibold rounded-md hover:bg-red-800 transition text-sm">
                                        Konfirmasi Hapus
                                    </button>
                                </div>
                            </div>
                        </div>
                        <?php endif; ?>

                    </div>
                </div>
            <?php endif; ?>
        </div>


        <!-- Tampilkan bagian lain hanya jika ada kelas yang dipilih -->
        <?php if ($current_class_id && count($students) > 0): ?>

        <!-- Tabel Daftar Hadir -->
        <div id="attendance-section">
            <h2 class="text-xl font-bold mb-4 text-gray-700">
                Data Kehadiran Harian
                <?php if (!$is_public_view && $is_admin_logged_in): ?>
                    <span class="text-base text-gray-500 font-normal">(Klik status untuk mengubah)</span>
                <?php endif; ?>
            </h2>
            <div class="overflow-x-auto shadow-lg rounded-xl border border-gray-200">
                <table class="min-w-full divide-y divide-gray-200 bg-white">
                    <thead class="bg-gray-100 sticky top-0 z-30">
                        <!-- Header Hari -->
                        <tr id="day-of-week-headers">
                             <th class="px-3 py-1 text-left text-xs font-medium text-gray-700 uppercase tracking-wider sticky-col bg-gray-100 min-w-[12rem] border-r border-gray-200"></th>
                            <?php foreach ($days_in_month as $day): ?>
                                <?php $isWeekend = $day['dayOfWeek'] === 0 || $day['dayOfWeek'] === 6; ?>
                                <th class="px-2 py-1 text-center text-xs font-medium uppercase tracking-wider min-w-[4rem] border-l border-gray-200 <?= $isWeekend ? 'bg-red-200 text-red-700' : 'text-gray-700' ?>">
                                    <?= $days_of_week[$day['dayOfWeek']] ?>
                                </th>
                            <?php endforeach; ?>
                        </tr>
                        <!-- Header Tanggal -->
                        <tr id="date-headers">
                            <th class="px-3 py-2 text-left text-xs font-bold text-gray-800 uppercase tracking-wider sticky-col bg-gray-200 min-w-[12rem] rounded-tl-xl border-r border-gray-200">
                                Nama Siswa
                            </th>
                            <?php foreach ($days_in_month as $day): ?>
                                <?php $isWeekend = $day['dayOfWeek'] === 0 || $day['dayOfWeek'] === 6; ?>
                                <th class="px-2 py-2 text-center text-xs font-bold uppercase tracking-wider min-w-[4rem] border-l border-gray-200 <?= $isWeekend ? 'bg-red-100 text-red-600' : 'text-gray-700 bg-gray-200' ?>">
                                    <?= $day['day'] ?>
                                </th>
                            <?php endforeach; ?>
                        </tr>
                    </thead>
                    <tbody id="attendance-body" class="divide-y divide-gray-100">
                        <?php foreach ($students as $index => $student): 
                            $row_class = $index % 2 === 0 ? 'bg-white hover:bg-gray-50' : 'bg-gray-50 hover:bg-gray-100';
                        ?>
                            <tr class="<?= $row_class ?>">
                                <!-- Kolom Nama Siswa (Sticky) -->
                                <td class="px-3 py-2 whitespace-nowrap text-sm font-medium text-gray-900 sticky-col bg-inherit border-r border-gray-200">
                                    <?= htmlspecialchars($student['name']) ?>
                                </td>
                                <!-- Kolom Kehadiran Harian -->
                                <?php foreach ($days_in_month as $day): 
                                    $date_key = $day['full'];
                                    $current_status = $attendance_data_month[$student['id']][$date_key] ?? 'A';
                                ?>
                                    <td class="px-1 py-1 whitespace-nowrap text-sm text-gray-500 text-center border-l border-gray-100">
                                        <?php if (!$is_public_view && $is_admin_logged_in): ?>
                                            <!-- Mode Admin: Pakai Select Interaktif -->
                                            <select 
                                                class="attendance-select <?= getStatusClass($current_status, false) ?>"
                                                data-student-id="<?= htmlspecialchars($student['id']) ?>"
                                                data-date="<?= htmlspecialchars($date_key) ?>"
                                                onchange="handleAttendanceChange(this)"
                                            >
                                                <?php foreach ($status_options as $value => $label): ?>
                                                    <option value="<?= $value ?>" <?= $value === $current_status ? 'selected' : '' ?>>
                                                        <?= $value ?>
                                                    </option>
                                                <?php endforeach; ?>
                                            </select>
                                        <?php else: ?>
                                            <!-- Mode Publik: Pakai Span Read-Only -->
                                            <span class="inline-block <?= getStatusClass($current_status, true) ?>">
                                                <?= htmlspecialchars($current_status) ?>
                                            </span>
                                        <?php endif; ?>
                                    </td>
                                <?php endforeach; ?>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>

        <!-- Rekapitulasi & Export -->
        <div id="recap-section" class="mt-12">
            <h2 class="text-xl font-bold mb-4 text-gray-700">
                Rekapitulasi Bulanan
            </h2>

            <?php if (!$is_public_view && $is_admin_logged_in): ?>
            <div class="flex justify-end mb-4">
                <button id="export-csv-btn" class="flex items-center space-x-2 px-4 py-2 bg-purple-600 text-white font-semibold rounded-lg shadow-md hover:bg-purple-700 transition">
                    <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 10v6m0 0l-3-3m3 3l3-3m2 8H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"></path></svg>
                    <span>Ekspor Rekap CSV</span>
                </button>
            </div>
            <?php endif; ?>

            <div class="overflow-x-auto">
                <table id="recap-table" class="min-w-full divide-y divide-gray-200 shadow-lg rounded-xl border border-gray-200 bg-white">
                    <thead class="bg-gray-100">
                        <tr>
                            <th class="px-6 py-3 text-left text-xs font-bold text-gray-700 uppercase tracking-wider sticky left-0 bg-gray-100 min-w-[12rem] rounded-tl-xl">Nama Siswa</th>
                            <th class="px-6 py-3 text-center text-xs font-bold text-green-700 uppercase tracking-wider">Hadir (H)</th>
                            <th class="px-6 py-3 text-center text-xs font-bold text-red-700 uppercase tracking-wider">Sakit (S)</th>
                            <th class="px-6 py-3 text-center text-xs font-bold text-yellow-700 uppercase tracking-wider">Izin (I)</th>
                            <th class="px-6 py-3 text-center text-xs font-bold text-gray-700 uppercase tracking-wider">Alfa (A)</th>
                            <th class="px-6 py-3 text-center text-xs font-bold text-indigo-700 uppercase tracking-wider">Total Hari</th>
                            <th class="px-6 py-3 text-center text-xs font-bold text-indigo-700 uppercase tracking-wider">Persentase H (%)</th>
                        </tr>
                    </thead>
                    <tbody id="recap-body" class="divide-y divide-gray-100">
                        <?php 
                        $totalH = 0; $totalS = 0; $totalI = 0; $totalA = 0;
                        $totalDays = count($days_in_month);
                        
                        foreach ($students as $index => $student):
                            $counts = ['H' => 0, 'S' => 0, 'I' => 0, 'A' => 0];
                            $student_records = $attendance_data_month[$student['id']] ?? [];
                            
                            foreach ($student_records as $status) {
                                if (isset($counts[$status])) {
                                    $counts[$status]++;
                                }
                            }

                            $studentTotalAttendance = array_sum($counts);
                            $percentage = $totalDays > 0 ? number_format(($counts['H'] / $totalDays) * 100, 1) : 0.0;

                            $totalH += $counts['H'];
                            $totalS += $counts['S'];
                            $totalI += $counts['I'];
                            $totalA += $counts['A'];

                            $row_class = $index % 2 === 0 ? 'bg-white' : 'bg-gray-50';
                        ?>
                            <tr class="<?= $row_class ?>" data-student-name="<?= htmlspecialchars($student['name']) ?>">
                                <td class="px-6 py-3 whitespace-nowrap text-sm font-medium text-gray-900 sticky left-0 bg-inherit border-r border-gray-200"><?= htmlspecialchars($student['name']) ?></td>
                                <td class="px-6 py-3 whitespace-nowrap text-sm text-green-700 font-bold text-center" data-recap="H"><?= $counts['H'] ?></td>
                                <td class="px-6 py-3 whitespace-nowrap text-sm text-red-700 font-bold text-center" data-recap="S"><?= $counts['S'] ?></td>
                                <td class="px-6 py-3 whitespace-nowrap text-sm text-yellow-700 font-bold text-center" data-recap="I"><?= $counts['I'] ?></td>
                                <td class="px-6 py-3 whitespace-nowrap text-sm text-gray-700 font-bold text-center" data-recap="A"><?= $counts['A'] ?></td>
                                <td class="px-6 py-3 whitespace-nowrap text-sm text-indigo-700 font-bold text-center"><?= $studentTotalAttendance ?> / <?= $totalDays ?></td>
                                <td class="px-6 py-3 whitespace-nowrap text-sm text-indigo-700 font-bold text-center"><?= $percentage ?>%</td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                    <tfoot class="bg-indigo-100 font-bold text-gray-800">
                        <?php
                            $totalPossibleRecordsAvailable = count($students) * $totalDays;
                            $totalClassAttendance = $totalH + $totalS + $totalI + $totalA;
                            $classHPercentage = $totalPossibleRecordsAvailable > 0 ? number_format(($totalH / $totalPossibleRecordsAvailable) * 100, 1) : 0.0;
                        ?>
                        <tr id="class-summary">
                            <td class="px-6 py-3 text-left">Total Rekap Kelas</td>
                            <td class="px-6 py-3 text-center"><?= $totalH ?></td>
                            <td class="px-6 py-3 text-center"><?= $totalS ?></td>
                            <td class="px-6 py-3 text-center"><?= $totalI ?></td>
                            <td class="px-6 py-3 text-center"><?= $totalA ?></td>
                            <td class="px-6 py-3 text-center"><?= $totalClassAttendance ?> / <?= $totalPossibleRecordsAvailable ?></td>
                            <td class="px-6 py-3 text-center"><?= $classHPercentage ?>%</td>
                        </tr>
                    </tfoot>
                </table>
            </div>
        </div>
        <?php else: ?>
            <div class="text-center py-12 bg-white rounded-xl shadow-lg border border-gray-200">
                <p class="text-xl font-semibold text-gray-500">
                    <?php if ($current_class_id): ?>
                        Kelas <?= htmlspecialchars($current_class_name) ?> belum memiliki data siswa.
                    <?php else: ?>
                        Anda belum memiliki kelas. Silakan tambahkan kelas baru melalui panel Administrator.
                    <?php endif; ?>
                </p>
            </div>
        <?php endif; ?>
        
        <?php endif; ?> <!-- Tutup blok PHP 'else' dari KONDISI LOGIN -->
    </div>

    <!-- Script JavaScript (Hanya untuk Admin Mode yang terautentikasi) -->
    <?php if (!$is_public_view && $is_admin_logged_in): ?>
    <script>
        const CURRENT_CLASS_ID = '<?= htmlspecialchars($current_class_id ?? '') ?>';
        const CURRENT_CLASS_NAME = '<?= htmlspecialchars($current_class_name) ?>';
        const CURRENT_MONTH_YEAR = '<?= htmlspecialchars($current_month_year) ?>';
        const PHP_HANDLER_URL = window.location.href.split('?')[0]; 
        
        /** Menampilkan pesan status server sementara */
        function showServerStatus(message, isSuccess = true) {
            const statusElement = document.getElementById('server-status');
            
            statusElement.textContent = message;
            statusElement.className = `fixed top-0 left-1/2 -translate-x-1/2 w-11/12 max-w-lg mt-4 p-3 font-medium rounded-lg border transition duration-500 ease-in-out z-50 shadow-xl ${isSuccess ? 'bg-green-100 text-green-700 border-green-300' : 'bg-red-100 text-red-700 border-red-300'}`;
            statusElement.style.display = 'block';
            
            setTimeout(() => {
                statusElement.style.display = 'none';
            }, 4000);
        }

        /** Mengirim permintaan AJAX ke file PHP ini */
        async function sendAjaxRequest(data, successMessage, reload = false) {
            try {
                const formData = new URLSearchParams();
                for (const key in data) {
                    formData.append(key, data[key]);
                }
                
                const response = await fetch(PHP_HANDLER_URL, {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                    body: formData.toString()
                });

                const result = await response.json();
                
                if (result.success) {
                    showServerStatus(successMessage || result.message, true);
                    
                    if (reload === true) {
                        window.location.href = `${PHP_HANDLER_URL}?class_id=${result.new_id || CURRENT_CLASS_ID}`;
                    } else if (reload === 'hard') {
                         // Mengatur ulang tampilan untuk memuat data siswa/rekap baru
                         window.location.reload();
                    } else if (reload === 'delete_redirect') {
                        if (result.new_id) {
                            window.location.href = `${PHP_HANDLER_URL}?class_id=${result.new_id}`;
                        } else {
                            window.location.href = PHP_HANDLER_URL;
                        }
                    }
                } else {
                    showServerStatus(result.message, false);
                }

            } catch (error) {
                console.error("AJAX Gagal:", error);
                showServerStatus(`Koneksi atau Pemrosesan Gagal: ${error.message}`, false);
            }
        }
        
        /** Handler saat nilai dropdown kehadiran berubah */
        function handleAttendanceChange(selectElement) {
            const studentId = selectElement.dataset.studentId;
            const date = selectElement.dataset.date;
            const newStatus = selectElement.value;

            // Update class for visual feedback (CSS)
            selectElement.className = `attendance-select ${getStatusClass(newStatus)}`;

            // Kirim ke server
            sendAjaxRequest({
                action: 'update_attendance',
                class_id: CURRENT_CLASS_ID,
                student_id: studentId,
                date: date,
                status: newStatus
            }, 'Kehadiran berhasil disimpan.', 'hard'); 
        }

        // Fungsi JS untuk mendapatkan kelas CSS (harus sync dengan PHP)
        function getStatusClass(status) {
            switch (status) {
                case 'H': return 'H bg-green-100 text-green-700';
                case 'S': return 'S bg-red-100 text-red-700';
                case 'I': return 'I bg-yellow-100 text-yellow-700';
                case 'A': default: return 'A bg-gray-100 text-gray-700';
            }
        }

        // --- Fitur Baru: Ekspor Rekap ke CSV ---
        function exportRecapToCSV() {
            const table = document.getElementById('recap-table');
            if (!table) return showServerStatus("Tabel rekap tidak ditemukan.", false);

            const class_name = CURRENT_CLASS_NAME;
            const month = CURRENT_MONTH_YEAR;
            
            let csv = `Kelas: ${class_name}\nBulan: ${month}\n\n`;

            // 1. Ambil Header
            const headerRow = table.querySelector('thead tr:last-child');
            const headers = Array.from(headerRow.querySelectorAll('th')).map(th => th.textContent.trim().replace(/\s*\([^)]*\)\s*/g, ''));
            csv += headers.join(',') + '\n';

            // 2. Ambil Data Siswa
            const bodyRows = table.querySelectorAll('#recap-body tr');
            bodyRows.forEach(row => {
                const studentName = row.querySelector('td:first-child').textContent.trim();
                const counts = Array.from(row.querySelectorAll('td[data-recap]')).map(td => td.textContent.trim());
                const percentage = row.querySelector('td:last-child').textContent.trim();
                const total = row.querySelector('td:nth-last-child(2)').textContent.trim().split('/')[0].trim();
                
                // Urutan kolom: Nama, H, S, I, A, Total Hari, Persentase H
                let rowData = [
                    `"${studentName}"`, 
                    counts[0], // H
                    counts[1], // S
                    counts[2], // I
                    counts[3], // A
                    total, // Total Hari Terdata
                    percentage // Persentase
                ];

                csv += rowData.join(',') + '\n';
            });

            // 3. Buat file dan trigger download
            const blob = new Blob([csv], { type: 'text/csv;charset=utf-8;' });
            const link = document.createElement('a');
            const url = URL.createObjectURL(blob);
            
            link.setAttribute('href', url);
            link.setAttribute('download', `rekap_kehadiran_${class_name}_${month}.csv`);
            link.style.visibility = 'hidden';
            document.body.appendChild(link);
            link.click();
            document.body.removeChild(link);

            showServerStatus("Rekapitulasi berhasil diunduh!", true);
        }

        // --- Event Listeners JS ---
        window.onload = () => {
            // Togle Panel Admin
            const toggleButton = document.getElementById('toggle-admin-panel');
            const adminPanel = document.getElementById('admin-panel');
            const arrowIcon = document.getElementById('arrow-icon');
            
            if (toggleButton && adminPanel && arrowIcon) {
                toggleButton.addEventListener('click', () => {
                    const isHidden = adminPanel.classList.toggle('hidden');
                    if (isHidden) {
                        arrowIcon.classList.remove('rotate-180');
                    } else {
                        arrowIcon.classList.add('rotate-180');
                    }
                });
            }

            // 1. Tambah Kelas
            const addClassBtn = document.getElementById('add-class-btn');
            if (addClassBtn) {
                addClassBtn.addEventListener('click', () => {
                    const newName = document.getElementById('new-class-name').value.trim();
                    if (!newName) {
                        return showServerStatus('Nama kelas tidak boleh kosong.', false);
                    }
                    sendAjaxRequest({
                        action: 'add_class',
                        new_name: newName
                    }, 'Kelas berhasil ditambahkan. Memuat ulang halaman...', true); 
                });
            }

            // 2. Tambah Siswa Tunggal
            const addStudentBtn = document.getElementById('add-student-btn');
            if (addStudentBtn) {
                addStudentBtn.addEventListener('click', () => {
                    const name = document.getElementById('new-student-name').value.trim();
                    const gender = document.getElementById('new-student-gender').value;
                    const idNumber = document.getElementById('new-student-id-number').value.trim();

                    if (!name) {
                        return showServerStatus('Nama siswa wajib diisi.', false);
                    }
                    
                    sendAjaxRequest({
                        action: 'add_student',
                        class_id: CURRENT_CLASS_ID,
                        name: name,
                        gender: gender,
                        id_number: idNumber
                    }, 'Siswa berhasil ditambahkan. Memuat ulang tampilan...', 'hard');
                });
            }

            // 3. Simpan Siswa dari JSON Massal
            const saveStudentsBtn = document.getElementById('save-students-btn');
            if (saveStudentsBtn) {
                saveStudentsBtn.addEventListener('click', () => {
                    const jsonInput = document.getElementById('student-json-input').value;
                    sendAjaxRequest({
                        action: 'save_students_json',
                        class_id: CURRENT_CLASS_ID,
                        json_data: jsonInput
                    }, 'Daftar siswa berhasil diperbarui. Memuat ulang tampilan...', 'hard');
                });
            }

            // 4. Hapus Kelas (Dilindungi Password)
            const showDeleteBtn = document.getElementById('show-delete-area-btn');
            const deleteArea = document.getElementById('delete-verification-area');
            const confirmDeleteBtn = document.getElementById('confirm-delete-btn');
            const deletePasswordInput = document.getElementById('delete-password-input');

            if (showDeleteBtn && deleteArea && deletePasswordInput) {
                showDeleteBtn.addEventListener('click', () => {
                    // Toggle tampilan, tetapi jika sudah terlihat, kosongkan input
                    deleteArea.classList.toggle('hidden');
                    if (!deleteArea.classList.contains('hidden')) {
                         deletePasswordInput.value = ''; 
                    }
                });
            }

            if (confirmDeleteBtn && deletePasswordInput) {
                confirmDeleteBtn.addEventListener('click', () => {
                    const password = deletePasswordInput.value;
                    if (!password) {
                        return showServerStatus('Kata sandi wajib diisi untuk menghapus kelas.', false);
                    }
                    
                    sendAjaxRequest({
                        action: 'delete_class',
                        class_id: CURRENT_CLASS_ID,
                        password_check: password
                    }, 'Kelas berhasil dihapus. Mengalihkan...', 'delete_redirect');
                });
            }

            // 5. Ekspor Rekap CSV
            const exportCsvBtn = document.getElementById('export-csv-btn');
            if (exportCsvBtn) {
                exportCsvBtn.addEventListener('click', exportRecapToCSV);
            }
        }; // Penutup window.onload
    </script>
    <?php endif; ?> <!-- Penutup blok PHP 'if' untuk skrip admin -->
    
    <!-- Penanda Akhir File untuk membantu debugging -->
</body>
</html>