<?php
// functions.php

// --- FUNGSI-FUNGSI DATABASE ---

/**
 * Mengambil data sekolah berdasarkan ID.
 * @param mysqli $conn Koneksi database.
 * @param int $id ID sekolah.
 * @return array|null Data sekolah dalam bentuk array asosiatif, atau null jika tidak ditemukan.
 */
function getSchool($conn, $id) {
    $stmt = $conn->prepare("SELECT * FROM schools WHERE id = ?");
    $stmt->bind_param("i", $id);
    $stmt->execute();
    return $stmt->get_result()->fetch_assoc();
}

/**
 * Mengambil data kelas berdasarkan ID, beserta nama sekolahnya.
 * Menggunakan LEFT JOIN untuk memastikan data kelas tetap diambil meskipun sekolah terkait tidak ditemukan.
 * @param mysqli $conn Koneksi database.
 * @param int $id ID kelas.
 * @return array|null Data kelas dalam bentuk array asosiatif, atau null jika tidak ditemukan.
 */
function getClass($conn, $id) {
    $stmt = $conn->prepare("SELECT c.*, s.name as school_name FROM classes c LEFT JOIN schools s ON c.school_id = s.id WHERE c.id = ?");
    $stmt->bind_param("i", $id);
    $stmt->execute();
    return $stmt->get_result()->fetch_assoc();
}

/**
 * Mengambil data siswa berdasarkan ID, beserta nama kelas dan KKM-nya.
 * Menggunakan LEFT JOIN untuk memastikan data siswa tetap diambil meskipun kelas terkait tidak ditemukan.
 * @param mysqli $conn Koneksi database.
 * @param int $id ID siswa.
 * @return array|null Data siswa dalam bentuk array asosiatif, atau null jika tidak ditemukan.
 */
function getStudent($conn, $id) {
    // --- PERBAIKAN: Diubah dari JOIN menjadi LEFT JOIN untuk robustness ---
    $stmt = $conn->prepare("SELECT st.*, c.name as class_name, c.kktp FROM students st LEFT JOIN classes c ON st.class_id = c.id WHERE st.id = ?");
    // --------------------------------------------------------------------
    $stmt->bind_param("i", $id);
    $stmt->execute();
    return $stmt->get_result()->fetch_assoc();
}

/**
 * Mengambil data konfigurasi asesmen berdasarkan ID kelas.
 * @param mysqli $conn Koneksi database.
 * @param int $class_id ID kelas.
 * @return array|null Data asesmen dalam bentuk array asosiatif, atau null jika tidak ditemukan.
 */
function getAssessment($conn, $class_id) {
    $stmt = $conn->prepare("SELECT * FROM assessments WHERE class_id = ?");
    $stmt->bind_param("i", $class_id);
    $stmt->execute();
    return $stmt->get_result()->fetch_assoc();
}


// --- ALGORITMA PERHITUNGAN ---

/**
 * Menghitung nilai akhir siswa berdasarkan jawaban global dan nilai SLM per bab.
 * @param array $assessment_config Konfigurasi asesmen (kktp, total soal, detail bab & TP).
 * @param array $student_answers Jawaban benar siswa (pg, isian, essay).
 * @param array $slm_scores Nilai SLM/UH per bab.
 * @param array &$chapters_with_scores Referensi ke array yang akan diisi dengan nilai per bab dan TP.
 * @return float Nilai rata-rata akhir dari semua bab.
 */
function calculateFinalScores($assessment_config, $student_answers, $slm_scores, &$chapters_with_scores) {
    $kktp = $assessment_config['kktp'];
    $pg_total_soal = $assessment_config['pg_total'];
    $isian_total_soal = $assessment_config['isian_total'];
    $essay_total_soal = $assessment_config['essay_total'];
    $pg_benar = $student_answers['pg_correct'];
    $isian_benar = $student_answers['isian_correct'];
    $essay_benar = $student_answers['essay_correct'];

    // Hitung total TP untuk pembagian soal
    $total_tp_count = 0;
    foreach ($assessment_config['chapters'] as $chapter) {
        $total_tp_count += count($chapter['tps']);
    }
    if ($total_tp_count === 0) return 0;

    // Hitung tingkat pencapaian untuk setiap jenis soal
    $pg_ach_rate = ($pg_total_soal > 0) ? $pg_benar / $pg_total_soal : 0;
    $isian_ach_rate = ($isian_total_soal > 0) ? $isian_benar / $isian_total_soal : 0;
    $essay_ach_rate = ($essay_total_soal > 0) ? $essay_benar / $essay_total_soal : 0;

    // Distribusikan soal secara merata ke setiap TP
    $pg_max_per_tp = ($total_tp_count > 0) ? floor($pg_total_soal / $total_tp_count) : 0;
    $isian_max_per_tp = ($total_tp_count > 0) ? floor($isian_total_soal / $total_tp_count) : 0;
    $essay_max_per_tp = ($total_tp_count > 0) ? floor($essay_total_soal / $total_tp_count) : 0;

    // Bagikan sisa soal (jika ada) ke TP pertama
    $pg_remainder = $pg_total_soal - ($pg_max_per_tp * $total_tp_count);
    $isian_remainder = $isian_total_soal - ($isian_max_per_tp * $total_tp_count);
    $essay_remainder = $essay_total_soal - ($essay_max_per_tp * $total_tp_count);

    $tp_counter = 0;
    $total_final_bab_score_sum = 0;

    // Proses setiap bab
    foreach ($assessment_config['chapters'] as $chapter_index => &$chapter) {
        $chapter_id = $chapter['id'];
        $bab_score_sum_tp_normalized = 0;
        
        // Proses setiap TP dalam bab
        foreach ($chapter['tps'] as $tp_index => &$tp) {
            $tp_counter++;
            $tp_pg_max = $pg_max_per_tp + ($tp_counter <= $pg_remainder ? 1 : 0);
            $tp_isian_max = $isian_max_per_tp + ($tp_counter <= $isian_remainder ? 1 : 0);
            $tp_essay_max = $essay_max_per_tp + ($tp_counter <= $essay_remainder ? 1 : 0);

            $tp_pg_achieved = $tp_pg_max * $pg_ach_rate;
            $tp_isian_achieved = $tp_isian_max * $isian_ach_rate;
            $tp_essay_achieved = $tp_essay_max * $essay_ach_rate;

            $total_achieved_raw = $tp_pg_achieved + $tp_isian_achieved + $tp_essay_achieved;
            $total_max_raw = $tp_pg_max + $tp_isian_max + $tp_essay_max;

            // Normalisasi nilai TP ke skala 100
            $nilai_akhir_tp_100 = ($total_max_raw > 0) ? ($total_achieved_raw / $total_max_raw) * 100 : 0;
            $tp['score_100'] = $nilai_akhir_tp_100;
            $bab_score_sum_tp_normalized += $nilai_akhir_tp_100;
        }
        unset($tp); // Hapus referensi untuk keamanan

        // Hitung nilai rata-rata TP per bab
        $bab_score_normalized_tp = (count($chapter['tps']) > 0) ? $bab_score_sum_tp_normalized / count($chapter['tps']) : 0;
        $slm_score = $slm_scores[$chapter_id] ?? 0;
        
        // Hitung nilai akhir bab (rata-rata nilai TP dan nilai SLM)
        $final_bab_score = ($slm_score > 0) ? ($bab_score_normalized_tp + $slm_score) / 2 : $bab_score_normalized_tp;
        $chapter['final_score'] = $final_bab_score;
        $chapter['status'] = ($final_bab_score >= $kktp) ? 'KOMPETEN' : 'BELUM KOMPETEN';
        
        $chapters_with_scores[$chapter_id] = $chapter;
        $total_final_bab_score_sum += $final_bab_score;
    }
    unset($chapter); // Hapus referensi untuk keamanan

    // Kembalikan nilai rata-rata akhir dari semua bab
    return (count($assessment_config['chapters']) > 0) ? $total_final_bab_score_sum / count($assessment_config['chapters']) : 0;
}